/////////////////////////////////////////////////////////////////
//
// Subclasser.c
//
// Christian Schubert, 11/11/2003
// christian@cschubert.net
//
// Functions to map messages
//
/////////////////////////////////////////////////////////////////

#include <windows.h>

#define SUBCLASS_FLAG_INSTEAD 1
#define SUBCLASS_FLAG_BEFORE 2
#define SUBCLASS_FLAG_AFTER 3


// name of window property used here
#define PROPNAME "MySubClass"


// struct type holding the subclass info
typedef struct subclassinfo_tag
{
	UINT		nOrgMsg;
	HWND		hWndNew;
	UINT		nNewMsg;
	WORD		nFlags;
} subclassinfo;

typedef struct subclassinfobase_tag
{
	WORD			nMaxCount;
	WNDPROC			hWndProc;
	subclassinfo	subclassinfos[1];
} subclassinfobase;


// DLLMain
BOOL APIENTRY DllMain( HANDLE hModule, 
                       DWORD  ul_reason_for_call, 
                       LPVOID lpReserved
					 )
{
    switch (ul_reason_for_call)
	{
		case DLL_PROCESS_ATTACH:
		case DLL_THREAD_ATTACH:
		case DLL_THREAD_DETACH:
		case DLL_PROCESS_DETACH:
			break;
    }
    return TRUE;
}


// New window procedure
LRESULT CALLBACK NewWindowProc(
  HWND hwnd,      // handle to window
  UINT uMsg,      // message identifier
  WPARAM wParam,  // first message parameter
  LPARAM lParam   // second message parameter
)
{
	// get subclass info
	subclassinfobase* pscinfobase;
	WORD nCount;
	LRESULT nReturn1, nReturn2;

	// get property
	pscinfobase = (subclassinfobase*)GetProp ( hwnd, PROPNAME );

	if ( pscinfobase )
	{
		// find the message
		nCount = 0;
		while ( nCount < pscinfobase->nMaxCount )
		{
			// if subclassed message matches or general broadcasting is on
			if ( uMsg == pscinfobase->subclassinfos[nCount].nOrgMsg || 
				pscinfobase->subclassinfos[nCount].nOrgMsg == 0xFFFFFFFF )
			{
				if ( pscinfobase->subclassinfos[nCount].nFlags == SUBCLASS_FLAG_AFTER )
					// call original handler
					nReturn1 = CallWindowProc ( pscinfobase->hWndProc, hwnd, uMsg, wParam, lParam );

				// send message to replacement handler
				nReturn2 = SendMessage ( pscinfobase->subclassinfos[nCount].hWndNew, 
					(pscinfobase->subclassinfos[nCount].nOrgMsg == 0xFFFFFFFF)?
					uMsg + pscinfobase->subclassinfos[nCount].nNewMsg:
					pscinfobase->subclassinfos[nCount].nNewMsg, wParam, lParam );

				if ( pscinfobase->subclassinfos[nCount].nFlags == SUBCLASS_FLAG_BEFORE )
					// call original handler
					nReturn1 = CallWindowProc ( pscinfobase->hWndProc, hwnd, uMsg, wParam, lParam );

				// eat up the message?
				if ( pscinfobase->subclassinfos[nCount].nFlags == SUBCLASS_FLAG_INSTEAD )
					// yes, return value from replacement handler
					return nReturn2;
				else
					// return original handler return value
					return nReturn1;
			}

			nCount++;
		}
		// nothing to do, call the original window procedure
		return CallWindowProc ( pscinfobase->hWndProc, hwnd, uMsg, wParam, lParam );

	}

	// subclassing is not active
	return DefWindowProc ( hwnd, uMsg, wParam, lParam );
};


// Register new window procedure
__declspec ( dllexport ) BOOL RegSubclass ( HWND hWnd, UINT nOrgMsg, HWND hWndNew, UINT nNewMsg, WORD nFlags )
{
	subclassinfobase* pscinfobase;
	WORD nCount, nSlot;

	// do some minor checks to prevent infinite loops
	if ( hWnd == hWndNew && ( nOrgMsg == nNewMsg || ( nOrgMsg == 0xFFFF ) ) )
		return FALSE;

	// is window already subclassed by us?
	pscinfobase = (subclassinfobase*)GetProp ( hWnd, PROPNAME );
	if ( !pscinfobase )
	{
		// get some mem
		pscinfobase = (subclassinfobase*)HeapAlloc ( GetProcessHeap(), HEAP_ZERO_MEMORY, sizeof ( subclassinfobase ) );
		if ( !pscinfobase )
			return FALSE;
		// set struct data
		pscinfobase->nMaxCount = 1;
		pscinfobase->hWndProc = (WNDPROC) GetWindowLong ( hWnd, GWL_WNDPROC );
		pscinfobase->subclassinfos[0].nOrgMsg = nOrgMsg;
		pscinfobase->subclassinfos[0].hWndNew = hWndNew;
		pscinfobase->subclassinfos[0].nNewMsg = nNewMsg;
		pscinfobase->subclassinfos[0].nFlags = nFlags;
		// set window property with pointer to struct
		SetProp ( hWnd, PROPNAME, (HANDLE)pscinfobase );
		// set new window procedure
		SetWindowLong ( hWnd, GWL_WNDPROC, (long)NewWindowProc );
	}
	else
	{
		// see if there's a free slot
		nCount = 0;
		nSlot = 0xFFFF;
		while ( nCount < pscinfobase->nMaxCount )
		{
			// remember the index of a free slot
			if ( pscinfobase->subclassinfos[nCount].nOrgMsg == 0 )
				nSlot = nCount;

			// see if message is already registered
			if ( pscinfobase->subclassinfos[nCount].nOrgMsg == nOrgMsg )
			{
				// reuse the slot
				nSlot = nCount;
				return TRUE;
			}
				
			nCount++;
		}

		if ( nSlot != 0xFFFF )
		{
			// either we found the message already registered or a free slot was available	
			pscinfobase->subclassinfos[nCount].nOrgMsg = nOrgMsg;
			pscinfobase->subclassinfos[nCount].hWndNew = hWndNew;
			pscinfobase->subclassinfos[nCount].nNewMsg = nNewMsg;
			pscinfobase->subclassinfos[nCount].nFlags = nFlags;
			
			return TRUE;
		}

		// No free slot, let's get a new one
		pscinfobase = (subclassinfobase*)HeapReAlloc ( GetProcessHeap(), 0, pscinfobase, 
			sizeof ( subclassinfobase ) + (pscinfobase->nMaxCount) * sizeof ( subclassinfo ) );
		if ( !pscinfobase )
			return FALSE;
		// set struct data
		pscinfobase->subclassinfos[pscinfobase->nMaxCount].nOrgMsg = nOrgMsg;
		pscinfobase->subclassinfos[pscinfobase->nMaxCount].hWndNew = hWndNew;
		pscinfobase->subclassinfos[pscinfobase->nMaxCount].nNewMsg = nNewMsg;
		pscinfobase->subclassinfos[pscinfobase->nMaxCount].nFlags = nFlags;
		pscinfobase->nMaxCount++;

		// memory pointer could has changed due to HeapReAlloc, set the property again
		SetProp ( hWnd, PROPNAME, (HANDLE)pscinfobase );
	}

	return TRUE;

};

// Unregister new window procedure
__declspec ( dllexport ) BOOL UnregSubclass ( HWND hWnd, UINT nMsg )
{
	WORD nCount, nCountFreeSlots;

	// is window subclassed by us?
	subclassinfobase* pscinfobase = (subclassinfobase*)GetProp ( hWnd, PROPNAME );
	if ( pscinfobase )
	{
		// find the message
		if ( nMsg != 0xFFFFFFFF )
		{
			nCount = 0;
			nCountFreeSlots = 0;
			while ( nCount < pscinfobase->nMaxCount )
			{
				// count free slots
				if ( pscinfobase->subclassinfos[nCount].nOrgMsg == 0 )
					nCountFreeSlots++;

				if ( pscinfobase->subclassinfos[nCount].nOrgMsg == nMsg )
				{
					// mark the slot as unused
					pscinfobase->subclassinfos[nCount].nOrgMsg = 0;
					break;
				}

				nCount++;
			}

			if ( nCount >= pscinfobase->nMaxCount )
				// message not found
				return FALSE;
		}

		// check if all slots are free (or should be freed), then remove subclassing
		if ( pscinfobase->nMaxCount == nCountFreeSlots + 1 || nMsg == 0xFFFFFFFF )
		{
			// reset window procedure
			SetWindowLong ( hWnd, GWL_WNDPROC, (long)pscinfobase->hWndProc );
			// delete window property
			RemoveProp ( hWnd, PROPNAME );
			// free mem
			if ( !HeapFree ( GetProcessHeap(), 0, pscinfobase ) )
				return FALSE;
		}
	}

	return TRUE;
};

